using System;
using System.Data;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
using Method = System.Reflection.MethodBase;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for CTRatioReport.
	/// </summary>
	public class CTRatioReport:VbecsReport
	{
		private DataTable _detailedReport;
		private DataTable _physicianSummaryReport;
		private DataTable _specialtySummaryReport;
        private DataSet _dsReport;

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5386"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>CTRatioReport object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5387"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty CT Ratio Report constructor
		/// </summary>
		public CTRatioReport()
		{
			this.IncludeDetailedReport = true;
			this.IncludePhysicianSummary = false;
			this.IncludeSpecialtySummary = false;
		}

		///<Developers>
		///	<Developer>EDS EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7440"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7441"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CT Ratio Report constructor
		/// </summary>
		/// <param name="startDate">Start Date</param>
		/// <param name="endDate">End Date</param>
		/// <param name="isMultidivisional">Is Multi-divisional report</param>
		/// <param name="isScheduled">Is scheduled report</param>
		public CTRatioReport(DateTime startDate, DateTime endDate, bool isMultidivisional, bool isScheduled):base(startDate, endDate, isMultidivisional, isScheduled)
		{
			this.IncludeDetailedReport = true;
			this.IncludePhysicianSummary = false;
			this.IncludeSpecialtySummary = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5389"> 
		///		<ExpectedInput>Valid detailed report flag</ExpectedInput>
		///		<ExpectedOutput>Updated CTRatioReport object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5390"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether detailed report should be printed
		/// </summary>
		public bool IncludeDetailedReport
		{
			get
			{
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IncludeDetailedReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IncludeDetailedReport] = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5392"> 
		///		<ExpectedInput>Valid physician summary flag</ExpectedInput>
		///		<ExpectedOutput>Updated CTRatioReport object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5393"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether physician summary report should be printed
		/// </summary>
		public bool IncludePhysicianSummary
		{
			get
			{
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IncludePhysicianSummary]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IncludePhysicianSummary] = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5395"> 
		///		<ExpectedInput>Valid specialty summary flag</ExpectedInput>
		///		<ExpectedOutput>Updated CTRatioReport object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5396"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether specialty summary report should be printed
		/// </summary>
		public bool IncludeSpecialtySummary
		{
			get
			{
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IncludeSpecialtySummary]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IncludeSpecialtySummary] = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5397"> 
		///		<ExpectedInput>Valid CTRatioReport object</ExpectedInput>
		///		<ExpectedOutput>Report criteria string for report footer</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5398"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets report criteria text for C:T Ratio report footer
		/// </summary>
		public string ReportCriteria
		{
			get
			{
				string criteria = "Start Date: "+Common.VBECSDateTime.FormatDateString(this.StartDate)+"\n";
				criteria+="End Date: "+Common.VBECSDateTime.FormatDateTimeString(this.EndDate)+"\n";
				criteria+="Including: ";
				if (this.IncludeDetailedReport)
				{
					if (this.IncludePhysicianSummary || this.IncludeSpecialtySummary)
					{
						criteria+="Detailed & ";
					}
					else
					{
						criteria+="Detailed ";
					}
				}
				if (this.IncludePhysicianSummary)
				{
					criteria+="Physician Summary ";
				}
				if (this.IncludeSpecialtySummary)
				{
					criteria+="Specialty Summary ";
				}
				if (this.IsMultiDivisional)
				{
					criteria+="Multi-divisional ";
				}
				criteria+="Report";
				return criteria;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6266"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6267"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///		Generates a CTRatioReport object.
		/// </summary>
		/// <returns></returns>
		public DataSet GenerateReport()
		{
			this.GetCTRatioReport();

			DataSet ctreportDataSet = new DataSet();
			ctreportDataSet.Tables.Add(this.SubReportSections.Copy());
			ctreportDataSet.Tables.Add(this.DetailedReport.Copy());
			//
			if (this.IncludePhysicianSummary)
			{
				ctreportDataSet.Tables.Add(this.PhysicianSummaryReport.Copy());
			}
			if (this.IncludeSpecialtySummary)
			{
				ctreportDataSet.Tables.Add(this.SpecialtySummaryReport.Copy());	
			}
#if DEBUG
			//Uncomment if you need to regenerate XML schema for this report
			//ctreportDataSet.Tables[0].WriteXml("e:\\temp\\CTRatioReportDataSet.xml", XmlWriteMode.WriteSchema);
            //ctreportDataSet.Tables[1].WriteXml("e:\\temp\\CTRatioDetailedDataSet.xml", XmlWriteMode.WriteSchema);
            //ctreportDataSet.Tables[2].WriteXml("e:\\temp\\CTRatioPhysicianSummaryDataSet.xml", XmlWriteMode.WriteSchema);
            //ctreportDataSet.Tables[2].WriteXml("e:\\temp\\CTRatioSpecialtySummaryDataSet.xml", XmlWriteMode.WriteSchema);
#endif
            return ctreportDataSet;
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7442"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7443"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate and print report
		/// Used by VbecsConsoleReport application
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{
            _dsReport = this.GenerateReport();
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(this.ReportCriteria);

            ReportDataSource reportSourceMain = new ReportDataSource();
            if (_dsReport.Tables.Count > 0)
            {
                reportSourceMain.Name = "CTRatioReportDataSet";
                reportSourceMain.Value = _dsReport.Tables[0];
            }

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.CTRatioReport.CTRatioReport.rdlc");
            Stream streamDetailed = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.CTRatioReport.CTRatioDetailed.rdlc");
            Stream streamPhysician = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.CTRatioReport.CTRatioPhysicianSummary.rdlc");
            Stream streamSpecialty = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.CTRatioReport.CTRatioSpecialtySummary.rdlc");

            LocalReport report = new LocalReport();
            
            report.SubreportProcessing += new SubreportProcessingEventHandler(LoadSubreportsEventHandler);

            report.DataSources.Add(reportSourceMain);
            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.LoadReportDefinition(stream);
            report.LoadSubreportDefinition("CTRatioDetailed", streamDetailed);
            report.LoadSubreportDefinition("CTRatioPhysicianSummary", streamPhysician);
            report.LoadSubreportDefinition("CTRatioSpecialtySummary", streamSpecialty);

            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
        }

        private void LoadSubreportsEventHandler(object sender, SubreportProcessingEventArgs e)
        {
            if (e.ReportPath == "CTRatioDetailed" || e.DataSourceNames[0].ToString() == "CTRatioDetailedDataSet")
            {
                ReportDataSource reportSourceDetailed = new ReportDataSource();
                if (_dsReport.Tables.Count > 1)
                {
                    reportSourceDetailed.Name = "CTRatioDetailedDataSet";
                    reportSourceDetailed.Value = _dsReport.Tables[1];
                }
                e.DataSources.Add(reportSourceDetailed);

                ReportDataSource reportSourceMain = new ReportDataSource();
                if (_dsReport.Tables.Count > 0)
                {
                    reportSourceMain.Name = "CTRatioReportDataSet";
                    reportSourceMain.Value = _dsReport.Tables[0];
                }
                e.DataSources.Add(reportSourceMain);
            }
            else if ((e.ReportPath == "CTRatioPhysicianSummary" || e.DataSourceNames[0].ToString() == "CTRatioPhysicianSummaryDataSet") && this.IncludePhysicianSummary)
            {
                ReportDataSource reportSourcePhysician = new ReportDataSource();
                if (_dsReport.Tables.Count > 2)
                {
                    reportSourcePhysician.Name = "CTRatioPhysicianSummaryDataSet";
                    reportSourcePhysician.Value = _dsReport.Tables[2];
                }
                e.DataSources.Add(reportSourcePhysician);

                ReportDataSource reportSourceMain = new ReportDataSource();
                if (_dsReport.Tables.Count > 0)
                {
                    reportSourceMain.Name = "CTRatioReportDataSet";
                    reportSourceMain.Value = _dsReport.Tables[0];
                }
                e.DataSources.Add(reportSourceMain);
            }
            else if ((e.ReportPath == "CTRatioSpecialtySummary" || e.DataSourceNames[0].ToString() == "CTRatioSpecialtySummaryDataSet") && this.IncludeSpecialtySummary)
            {
                ReportDataSource reportSourceSpecialty = new ReportDataSource();
                if (_dsReport.Tables.Count > 2)
                {
                    reportSourceSpecialty.Name = "CTRatioSpecialtySummaryDataSet";
                    reportSourceSpecialty.Value = _dsReport.Tables[2];
                }
                e.DataSources.Add(reportSourceSpecialty);

                ReportDataSource reportSourceMain = new ReportDataSource();
                if (_dsReport.Tables.Count > 0)
                {
                    reportSourceMain.Name = "CTRatioReportDataSet";
                    reportSourceMain.Value = _dsReport.Tables[0];
                }
                e.DataSources.Add(reportSourceMain);
            }
        }

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5399"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Count of records in detailed C:T ratio report</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5400"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a count of the number of records in the detailed C:T Ratio Report section, based on search criteria
		/// </summary>
		/// <returns>Record count</returns>
		public int GetCTRatioReportCount()
		{
			return DAL.CTRatioReport.GetCTRatioReportCount(this.StartDate, this.EndDate, Common.LogonUser.LogonUserDivisionCode, this.IsMultiDivisional);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5401"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>C:T ratio report data created</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5402"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Builds the C:T Ratio report and, if required, the summary report for SSRS based on report criteria
		/// </summary>
		public void GetCTRatioReport()
		{
			DataTable detailedReport = DAL.CTRatioReport.GetCTRatioReport(this.StartDate, this.EndDate, Common.LogonUser.LogonUserDivisionCode, this.IsMultiDivisional);
			detailedReport.Columns.Add(ARTIFICIAL.PatientDisplayName);
			detailedReport.Columns.Add(ARTIFICIAL.ProviderDisplayName);
			detailedReport.Columns.Add(ARTIFICIAL.NoDataIndicator,typeof(bool));
			//
			foreach(DataRow dr in detailedReport.Rows)
			{
				if (Convert.ToChar(dr[TABLE.UnitStatus.UnitStatusCode]) != Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Transfused))
				{
					dr[TABLE.UnitStatus.UnitStatusText] = "";
				}
				else 
				{
					//CR 2434 - check if the unit was issued to this patient and if yes mark it as transfused or presumed-transfused
					DataTable dtIssuedUnits = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid((Guid)dr[Common.VbecsTables.BloodUnit.BloodUnitGuid]);
					if (dtIssuedUnits.Select(Common.VbecsTables.Patient.PatientGuid + " = '" + dr[Common.VbecsTables.Patient.PatientGuid].ToString() + "'").Length > 0)
					{
						if (!dr.IsNull(TABLE.BloodUnitStatus.PresumedTransfusedIndicator))
						{
							if (Convert.ToBoolean(dr[TABLE.BloodUnitStatus.PresumedTransfusedIndicator]))
							{
								dr[TABLE.UnitStatus.UnitStatusText] = "Presumed";
							}
						}
					}
					else
					{
						dr[TABLE.UnitStatus.UnitStatusText] = "";
					}
				}
				//
				dr[ARTIFICIAL.PatientDisplayName] = BOL.Patient.BuildDisplayName(dr[TABLE.Patient.PatientFirstName].ToString(),dr[TABLE.Patient.PatientMiddleName].ToString(),dr[TABLE.Patient.PatientLastName].ToString());
				dr[ARTIFICIAL.ProviderDisplayName] = BOL.Patient.BuildDisplayName(dr[TABLE.PatientOrder.OrderingProviderFirstName].ToString(),dr[TABLE.PatientOrder.OrderingProviderMiddleInitial].ToString(),dr[TABLE.PatientOrder.OrderingProviderLastName].ToString());
				dr[ARTIFICIAL.NoDataIndicator] = false;
			}
			//
			if ((detailedReport.Rows.Count>0)&& this.IsMultiDivisional)
			{
				//Create a blank row for any division that does not have a record

				VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
				foreach(BOL.Division division in user.DivisionsAndRoles.EffectiveDivisions)
				{
					DataRow[] drDivision = detailedReport.Select(TABLE.VamcDivision.DivisionCode + " = '" + division.DivisionCode + "'");
					if (drDivision.Length == 0)
					{
						//
						foreach(DataColumn col in detailedReport.Columns)
						{
							col.AllowDBNull = true;
						}
						//this division is not represented in the report
						DataRow drBlank = detailedReport.NewRow();
						drBlank[TABLE.VamcDivision.DivisionCode] = division.DivisionCode;
						drBlank[TABLE.VamcDivision.DivisionName] = division.DivisionName;
						drBlank[ARTIFICIAL.NoDataIndicator] = true;
						detailedReport.Rows.Add(drBlank);
					}
				}
			}
			//
			this._detailedReport = BOL.VbecsReport.MultiDivisionalSort(detailedReport);
			//
			if (this.IncludePhysicianSummary)
			{
				GetPhysicianSummaryReport();
			}
			//
			if (this.IncludeSpecialtySummary)
			{
				GetSpecialtySummaryReport();
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///
		/// <summary>
		/// Creates table schema for physician summary report
		/// </summary>
		/// <returns>Physician Summary report schema</returns>
		private DataTable CreatePhysicianSummaryTable()
		{
			DataTable dt = new DataTable("PhysicianSummaryReport");
			dt.Columns.Add(TABLE.VamcDivision.DivisionCode);
			dt.Columns.Add(TABLE.VamcDivision.DivisionName);
			dt.Columns.Add(TABLE.PatientOrder.OrderingProviderId);
			dt.Columns.Add(ARTIFICIAL.ProviderDisplayName);
			dt.Columns.Add(ARTIFICIAL.Crossmatched,typeof(int));
			dt.Columns.Add(ARTIFICIAL.Transfused,typeof(int));
			dt.Columns.Add(ARTIFICIAL.NoDataIndicator, typeof(bool));
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///
		/// <summary>
		/// Builds the Phsyician Summary report, based on the detailed report info.
		/// </summary>
		private void GetPhysicianSummaryReport()
		{
			this._physicianSummaryReport = CreatePhysicianSummaryTable();
			//
			DataRow[] dtPhysicians = this._detailedReport.Select("","ThisDivision DESC, DivisionName ASC, ProviderDisplayName ASC");
			//
			string divisioncode = null;
			string physicianid = null;
			DataRow drNewEntry = null;
			//
			foreach(DataRow dr in dtPhysicians)
			{
				if ((divisioncode != dr[TABLE.VamcDivision.DivisionCode].ToString())||
					(physicianid != dr[TABLE.PatientOrder.OrderingProviderId].ToString()))
				{
					divisioncode = dr[TABLE.VamcDivision.DivisionCode].ToString();
					physicianid = dr[TABLE.PatientOrder.OrderingProviderId].ToString();
					//
					drNewEntry = _physicianSummaryReport.NewRow();
					drNewEntry[TABLE.VamcDivision.DivisionCode] = dr[TABLE.VamcDivision.DivisionCode];
					drNewEntry[TABLE.VamcDivision.DivisionName] = dr[TABLE.VamcDivision.DivisionName];
					if ((bool)dr[ARTIFICIAL.NoDataIndicator])
					{
						drNewEntry[ARTIFICIAL.NoDataIndicator] = true;
					}
					else
					{
						drNewEntry[TABLE.PatientOrder.OrderingProviderId] = dr[TABLE.PatientOrder.OrderingProviderId];
						drNewEntry[ARTIFICIAL.ProviderDisplayName] = dr[ARTIFICIAL.ProviderDisplayName];
						drNewEntry[ARTIFICIAL.NoDataIndicator] = false;
					}
					drNewEntry[ARTIFICIAL.Crossmatched] = 0;
					drNewEntry[ARTIFICIAL.Transfused] = 0;
					_physicianSummaryReport.Rows.Add(drNewEntry);
				}
				if ((bool)dr[ARTIFICIAL.NoDataIndicator] == false)
				{
                    //CR 3554
				    if (drNewEntry != null && drNewEntry[ARTIFICIAL.Crossmatched] != null)
				    {
                        drNewEntry[ARTIFICIAL.Crossmatched] = ((int)drNewEntry[ARTIFICIAL.Crossmatched]) + 1;
				    }
				    
					if (dr[TABLE.UnitStatus.UnitStatusCode].ToString() == "T")
					{
						//CR 2434 - check if the unit was issued to this patient and if yes mark it as transfused or presumed-transfused
						DataTable dtIssuedUnits = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid((Guid)dr[Common.VbecsTables.BloodUnit.BloodUnitGuid]);
						if (dtIssuedUnits.Select(Common.VbecsTables.Patient.PatientGuid + " = '" + dr[Common.VbecsTables.Patient.PatientGuid].ToString() + "'").Length > 0)
						{
                            //CR 3554
						    if (drNewEntry == null)
						        throw new ApplicationException("drNewEntry is null");
							drNewEntry[ARTIFICIAL.Transfused] = ((int)drNewEntry[ARTIFICIAL.Transfused])+1;
						}
					}
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///
		/// <summary>
		/// Creates table schema for specialty summary report
		/// </summary>
		/// <returns>Specialty Summary report schema</returns>
		private DataTable CreateSpecialtySummaryTable()
		{
			DataTable dt = new DataTable("SpecialtySummaryReport");
			dt.Columns.Add(TABLE.VamcDivision.DivisionCode);
			dt.Columns.Add(TABLE.VamcDivision.DivisionName);
			dt.Columns.Add(TABLE.TreatingSpecialty.TreatingSpecialtyCode);
			dt.Columns.Add(TABLE.TreatingSpecialty.TreatingSpecialtyName);
			dt.Columns.Add(ARTIFICIAL.Crossmatched,typeof(int));
			dt.Columns.Add(ARTIFICIAL.Transfused,typeof(int));
			dt.Columns.Add(ARTIFICIAL.NoDataIndicator, typeof(bool));
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///
		/// <summary>
		/// Builds the Specialty Summary report, based on the detailed report info.
		/// </summary>
		private void GetSpecialtySummaryReport()
		{
			this._specialtySummaryReport = CreateSpecialtySummaryTable();
			//
			DataRow[] dtSpecialties = this._detailedReport.Select("","ThisDivision DESC, DivisionName ASC, SortColumn ASC");
			//
			string divisioncode = null;
			string specialtyid = null;
			DataRow drNewEntry = null;
			//
			foreach(DataRow dr in dtSpecialties)
			{
				if ((divisioncode != dr[TABLE.VamcDivision.DivisionCode].ToString())||
					(specialtyid != dr[TABLE.TreatingSpecialty.TreatingSpecialtyCode].ToString()))
				{
					divisioncode = dr[TABLE.VamcDivision.DivisionCode].ToString();
					specialtyid = dr[TABLE.TreatingSpecialty.TreatingSpecialtyCode].ToString();
					//
					drNewEntry = _specialtySummaryReport.NewRow();
					drNewEntry[TABLE.VamcDivision.DivisionCode] = dr[TABLE.VamcDivision.DivisionCode];
					drNewEntry[TABLE.VamcDivision.DivisionName] = dr[TABLE.VamcDivision.DivisionName];
					if ((bool)dr[ARTIFICIAL.NoDataIndicator])
					{
						drNewEntry[ARTIFICIAL.NoDataIndicator] = true;
					}
					else
					{
						drNewEntry[TABLE.TreatingSpecialty.TreatingSpecialtyCode] = dr[TABLE.TreatingSpecialty.TreatingSpecialtyCode];
						if (dr[TABLE.TreatingSpecialty.TreatingSpecialtyName].ToString().Length==0)
						{
							drNewEntry[TABLE.TreatingSpecialty.TreatingSpecialtyName] = "Unknown Specialty";
						}
						else
						{
							drNewEntry[TABLE.TreatingSpecialty.TreatingSpecialtyName] = dr[TABLE.TreatingSpecialty.TreatingSpecialtyName];
						}
						drNewEntry[ARTIFICIAL.NoDataIndicator] = false;
					}
					drNewEntry[ARTIFICIAL.Crossmatched] = 0;
					drNewEntry[ARTIFICIAL.Transfused] = 0;
					_specialtySummaryReport.Rows.Add(drNewEntry);
				}
				if ((bool)dr[ARTIFICIAL.NoDataIndicator] == false)
				{
                    //CR 3554
                    if (drNewEntry != null && drNewEntry[ARTIFICIAL.Crossmatched] != null)
                        drNewEntry[ARTIFICIAL.Crossmatched] = ((int)drNewEntry[ARTIFICIAL.Crossmatched]) + 1;
                        
					if (dr[TABLE.UnitStatus.UnitStatusCode].ToString() == "T")
					{
						//CR 2434 - check if the unit was issued to this patient and if yes mark it as transfused or presumed-transfused
                        //CR 3554 added checks for null
						DataTable dtIssuedUnits = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid((Guid)dr[Common.VbecsTables.BloodUnit.BloodUnitGuid]);
                        if (drNewEntry != null && drNewEntry[ARTIFICIAL.Transfused] != null && dtIssuedUnits.Select(Common.VbecsTables.Patient.PatientGuid + " = '" + dr[Common.VbecsTables.Patient.PatientGuid].ToString() + "'").Length > 0)
						{
							drNewEntry[ARTIFICIAL.Transfused] = ((int)drNewEntry[ARTIFICIAL.Transfused])+1;
						}
					}
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5403"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Detailed C:T Ratio report data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5404"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets data table containing detailed report information for SSRS sub-report
		/// </summary>
		public DataTable DetailedReport
		{
			get
			{
				if (this._detailedReport == null)
				{
					this.GetCTRatioReport();
				}
				return this._detailedReport;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5405"> 
		///		<ExpectedInput>Valid CTRatioReport object with IncludePhysicianSummary flag set to true</ExpectedInput>
		///		<ExpectedOutput>Physician Summary report data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5406"> 
		///		<ExpectedInput>Valid CTRatioReport object with IncludePhysicianSummary flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business Object excpetion thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets data table containing physician summary report information for SSRS sub-report
		/// </summary>
		public DataTable PhysicianSummaryReport
		{
			get
			{
				if (!this.IncludePhysicianSummary)
				{
					throw new BOL.BusinessObjectException("Physician Summary report was not requested");
				}
				if (this._detailedReport == null)
				{
					this.GetCTRatioReport();
				}
				return this._physicianSummaryReport;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5408"> 
		///		<ExpectedInput>Valid CTRatioReport object with IncludeSpecialtySummary flag set to true</ExpectedInput>
		///		<ExpectedOutput>Specialty Summary report data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5409"> 
		///		<ExpectedInput>Valid CTRatioReport object with IncludeSpecialtySummary flag set to false</ExpectedInput>
		///		<ExpectedOutput>Business Object excpetion thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets data table containing specialty summary report information for SSRS sub-report
		/// </summary>
		public DataTable SpecialtySummaryReport
		{
			get
			{
				if (!this.IncludeSpecialtySummary)
				{
					throw new BOL.BusinessObjectException("Specialty Summary report was not requested");
				}
				if (this._detailedReport == null)
				{
					this.GetCTRatioReport();
				}
				return this._specialtySummaryReport;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5411"> 
		///		<ExpectedInput>Valid CTRatioReport object</ExpectedInput>
		///		<ExpectedOutput>Data table detailing included sections and record counts returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5412"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets data table containing sub report section information for SSRS report
		/// </summary>
		/// <returns>Data table</returns>
		public DataTable SubReportSections
		{
			get
			{
				//
				if (this._detailedReport == null)
				{
					this.GetCTRatioReport();
				}
				//
				DataTable dtSections = new DataTable("Sections");
				dtSections.Columns.Add(ARTIFICIAL.IncludeDetailedReport,typeof(bool));
				dtSections.Columns.Add(ARTIFICIAL.DetailedReportRowCount,typeof(int));
				dtSections.Columns.Add(ARTIFICIAL.IncludePhysicianSummary,typeof(bool));
				dtSections.Columns.Add(ARTIFICIAL.PhysicianSummaryRowCount,typeof(int));
				dtSections.Columns.Add(ARTIFICIAL.IncludeSpecialtySummary,typeof(bool));
				dtSections.Columns.Add(ARTIFICIAL.SpecialtySummaryRowCount,typeof(int));
				//
				DataRow drSection = dtSections.NewRow();
				//
				if (this.IncludeDetailedReport)
				{
					drSection[ARTIFICIAL.IncludeDetailedReport] = true;
					drSection[ARTIFICIAL.DetailedReportRowCount] = 0;
					foreach(DataRow dr in this._detailedReport.Rows)
					{
						if ((bool)dr[ARTIFICIAL.NoDataIndicator] == false)
						{
							drSection[ARTIFICIAL.DetailedReportRowCount]=(int)drSection[ARTIFICIAL.DetailedReportRowCount]+1;
						}
					}
				}
				else
				{
					drSection[ARTIFICIAL.IncludeDetailedReport] = false;
					drSection[ARTIFICIAL.DetailedReportRowCount] = 0;
				}
				//
				if (this.IncludePhysicianSummary)
				{
					drSection[ARTIFICIAL.IncludePhysicianSummary] = true;
					drSection[ARTIFICIAL.PhysicianSummaryRowCount] = 0;
					foreach(DataRow dr in this._physicianSummaryReport.Rows)
					{
						if ((bool)dr[ARTIFICIAL.NoDataIndicator] == false)
						{
							drSection[ARTIFICIAL.PhysicianSummaryRowCount]=(int)drSection[ARTIFICIAL.PhysicianSummaryRowCount]+1;
						}
					}
				}
				else
				{
					drSection[ARTIFICIAL.IncludePhysicianSummary] = false;
					drSection[ARTIFICIAL.PhysicianSummaryRowCount] = 0;
				}
				//
				if (this.IncludeSpecialtySummary)
				{
					drSection[ARTIFICIAL.IncludeSpecialtySummary] = true;
					drSection[ARTIFICIAL.SpecialtySummaryRowCount] = 0;
					foreach(DataRow dr in this._specialtySummaryReport.Rows)
					{
						if ((bool)dr[ARTIFICIAL.NoDataIndicator] == false)
						{
							drSection[ARTIFICIAL.SpecialtySummaryRowCount]=(int)drSection[ARTIFICIAL.SpecialtySummaryRowCount]+1;
						}
					}
				}
				else
				{
					drSection[ARTIFICIAL.IncludeSpecialtySummary] = false;
					drSection[ARTIFICIAL.SpecialtySummaryRowCount] = 0;
				}
				//
				dtSections.Rows.Add(drSection);
				//
				return dtSections;
			}
		}
	}
}
